/*
  Copyright (C) 2013, 2024 Rocky Bernstein <rocky@gnu.org>
  Copyright (C) 2014 Robert Kausch <robert.kausch@freac.org>

  This program is free software: you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation, either version 3 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

/*
   Regression test for lib/cdda_interface/toc.c

   To compile as s standalone program:

 gcc -g3 -Wall -DHAVE_CONFIG_H -I../.. -I../..include toc.c \
      ../../lib/paranoia/libcdio_paranoia.la ../../lib/cdda_interface/.lib/libcdio_cdda.a -lcdio -o toc

*/

#ifdef HAVE_CONFIG_H
#include "config.h"
#define __CDIO_CONFIG_H__ 1
#endif

#ifdef HAVE_STDIO_H
#include <stdio.h>
#endif
#ifdef HAVE_STDLIB_H
#include <stdlib.h>
#endif
#ifdef HAVE_STRING_H
#include <string.h>
#endif

#include <cdio/paranoia/cdda.h>
#include <cdio/cd_types.h>
#include <cdio/logging.h>

#ifndef DATA_DIR
#define DATA_DIR "/src/external-vcs/github/rocky/libcdio-paranoia/../data"
#endif

static void
log_handler (cdio_log_level_t level, const char message[])
{
  switch(level) {
  case CDIO_LOG_DEBUG:
  case CDIO_LOG_INFO:
    return;
  default:
    printf("cdio %d message: %s\n", level, message);
  }
}

static int
cdda(cdrom_drive_t *d) {
    track_t t = 0;
    lsn_t lsn = 0;

    /* Check number of tracks.
     */
    t = cdda_tracks(d);
    if (t != 1) {
	printf("Should have found one track, got %d\n", t);
	return 1;
    }

    /* Check disc sector reporting.
     */
    lsn = cdda_disc_firstsector(d);
    if ( lsn != 0 ) {
	printf("Should have gotten 0 as the first sector, got %d.\n", lsn);
	return 2;
    }

    lsn = cdda_disc_lastsector(d);
    if ( lsn != 301 ) {
	printf("Should have gotten 301 as the last sector, got %d.\n", lsn);
	return 2;
    }

    /* Check pregap sector reporting (should fail here).
     */
    lsn = cdda_track_firstsector(d, 0);
    if ( lsn != -402 ) {
	printf("Should have gotten error code -402, got %d.\n", lsn);
	return 3;
    }

    /* Check track sector reporting.
     */
    lsn  = cdda_track_firstsector(d, 1);
    if ( lsn != 0 ) {
	printf("Should have gotten 0 as the first sector, got %d.\n", lsn);
	return 4;
    }

    lsn  = cdda_track_lastsector(d, 1);
    if ( lsn != 301 ) {
	printf("Should have gotten 301 as the last sector, got %d.\n", lsn);
	return 4;
    }

    /* Check CDIO_CDROM_LEADOUT_TRACK/tracks+1.
     */
    lsn  = cdda_track_firstsector(d, CDIO_CDROM_LEADOUT_TRACK);
    if ( lsn != 302 ) {
	printf("First: Should have gotten 302 as first sector on leadout, got %d.\n", lsn);
	return 5;
    }

    lsn  = cdda_track_firstsector(d, 2);
    if ( lsn != 302 ) {
	printf("First: Should have gotten 302 as first sector on leadout, got %d.\n", lsn);
	return 5;
    }

    lsn  = cdda_track_lastsector(d, CDIO_CDROM_LEADOUT_TRACK);
    if ( lsn != -401 ) {
	printf("Last: Should have gotten error code -401, got %d.\n", lsn);
	return 5;
    }

    lsn  = cdda_track_lastsector(d, 2);
    if ( lsn != -401 ) {
	printf("Last: Should have gotten error code -401, got %d.\n", lsn);
	return 5;
    }

    /* Check track ranges.
     */
    lsn  = cdda_track_firstsector(d, 3);
    if ( lsn != -401 ) {
	printf("Should have gotten error code -401, got %d.\n", lsn);
	return 6;
    }

    lsn  = cdda_track_lastsector(d, -1);
    if ( lsn != -401 ) {
	printf("Should have gotten error code -401, got %d.\n", lsn);
	return 6;
    }

    /* Check sector track reporting.
     */
    t = cdda_sector_gettrack(d, 100);
    if ( 1 != t )  {
	printf("Should have gotten 1 as the track, got %d.\n", t);
	return 7;
    }

    t = cdda_sector_gettrack(d, 1000);
    if ( CDIO_INVALID_TRACK != t )  {
	printf("Should have gotten CDIO_INVALID_TRACK as the track, got %d.\n", t);
	return 7;
    }

    return 0;
}

static int
cd_extra(cdrom_drive_t *d) {
    track_t t = 0;
    lsn_t lsn = 0;

    /* Check number of tracks (data track also counted).
     */
    t = cdda_tracks(d);
    if (t != 2) {
	printf("Should have found two tracks, got %d\n", t);
	return 1;
    }

    /* Check disc sector reporting.
     */
    lsn = cdda_disc_lastsector(d);
    if ( lsn != 149 ) {
	printf("Should have gotten 149 as the last sector, got %d.\n", lsn);
	return 2;
    }

    /* Check track sector reporting.
     */
    lsn  = cdda_track_lastsector(d, 1);
    if ( lsn != 149 ) {
	printf("Should have gotten 149 as the last sector, got %d.\n", lsn);
	return 4;
    }

    /* Check CDIO_CDROM_LEADOUT_TRACK/tracks+1.
     */
    lsn  = cdda_track_firstsector(d, 3);
    if ( lsn != 302 ) {
	printf("First: Should have gotten 302 as first sector on leadout, got %d.\n", lsn);
	return 5;
    }

    lsn  = cdda_track_lastsector(d, 3);
    if ( lsn != -401 ) {
	printf("Last: Should have gotten error code -401, got %d.\n", lsn);
	return 5;
    }

    /* Check track ranges.
     */
    lsn  = cdda_track_firstsector(d, 4);
    if ( lsn != -401 ) {
	printf("Should have gotten error code -401, got %d.\n", lsn);
	return 6;
    }

    /* Check non audio tracks.
     */
    lsn  = cdda_track_firstsector(d, 2);
    if ( lsn != 150 ) {
	printf("Should have gotten 150 as first sector, got %d.\n", lsn);
	return 8;
    }

    lsn  = cdda_track_lastsector(d, 2);
    if ( lsn != 301 ) {
	printf("Should have gotten 301 as last sector, got %d.\n", lsn);
	return 8;
    }

    return 0;
}

static int
hidden_track(cdrom_drive_t *d) {
    track_t t = 0;
    lsn_t lsn = 0;

    /* Check number of tracks (hidden track ignored).
     */
    t = cdda_tracks(d);
    if (t != 1) {
	printf("Should have found one track, got %d\n", t);
	return 1;
    }

    /* Check pregap sector reporting.
     */
    lsn = cdda_track_firstsector(d, 0);
    if ( lsn != 0 ) {
	printf("Should have gotten 0 as the first sector on hidden track, got %d.\n", lsn);
	return 2;
    }

    /* Check track sector reporting.
     */
    lsn  = cdda_track_firstsector(d, 1);
    if ( lsn != 150 ) {
	printf("Should have gotten 150 as the first sector, got %d.\n", lsn);
	return 3;
    }

    lsn  = cdda_track_lastsector(d, 1);
    if ( lsn != 301 ) {
	printf("Should have gotten 301 as the last sector, got %d.\n", lsn);
	return 3;
    }

    return 0;
}

static int
mixed_mode_cd(cdrom_drive_t *d) {
    track_t t = 0;
    lsn_t lsn = 0;

    /* Check number of tracks (data track also counted).
     */
    t = cdda_tracks(d);
    if (t != 2) {
	printf("Should have found two tracks, got %d\n", t);
	return 1;
    }

    /* Check disc sector reporting.
     */
    lsn = cdda_disc_firstsector(d);
    if ( lsn != 150 ) {
	printf("Should have gotten 150 as the first sector, got %d.\n", lsn);
	return 1;
    }

    lsn = cdda_disc_lastsector(d);
    if ( lsn != 301 ) {
	printf("Should have gotten 301 as the last sector, got %d.\n", lsn);
	return 1;
    }

    /* Check non audio tracks.
     */
    lsn  = cdda_track_firstsector(d, 1);
    if ( lsn != 0 ) {
	printf("Should have gotten 0 as the first sector, got %d.\n", lsn);
	return 8;
    }

    lsn  = cdda_track_lastsector(d, 1);
    if ( lsn != 149 ) {
	printf("Should have gotten 149 as the last sector, got %d.\n", lsn);
	return 8;
    }

    return 0;
}

static int
perform_test(const char *dir, const char *file, int (*func)(cdrom_drive_t *)){
  cdrom_drive_t *d = NULL; /* Place to store handle given by cd-paranoia. */
  CdIo_t *p_cdio;
  int rc;

  /* Below, +1 for null byte at the end */
  size_t drv_len = strlen(dir) + strlen(file) + strlen(".toc") + 1;
  char *drv = (char*) malloc(drv_len);
  strncpy(drv, dir, drv_len);
  strncat(drv, file, drv_len);

  if (cdio_have_driver(DRIVER_BINCUE)) {
    p_cdio = cdio_open(strncat(drv, ".cue", drv_len), DRIVER_UNKNOWN);
  } else if (cdio_have_driver(DRIVER_CDRDAO)) {
    p_cdio = cdio_open(strncat(drv, ".toc", drv_len), DRIVER_UNKNOWN);
  } else {
    printf("-- You don't have enough drivers for the test (%s).\n", file);
    free(drv);
    return 77;
  }

  d=cdio_cddap_identify_cdio(p_cdio, CDDA_MESSAGE_PRINTIT, NULL);

  if ( !d ) {
    printf("Should have identified as an audio CD disc (%s).\n", file);
    free(drv);
    return 1;
  }

  if ( 0 != cdio_cddap_open(d) ) {
    printf("Unable to open disc (%s).\n", file);
    free(drv);
    return 2;
  }

  rc = func(d);
  cdio_cddap_close_no_free_cdio(d);
  cdio_destroy(p_cdio);
	free(drv);

  return rc;
}

int
main(int argc, const char *argv[])
{
  int i=0;
  int rc;
  const char *files[] = { "cdda", "cd-extra", "hidden-track", "mixed-mode-cd", NULL };
  int (*funcs[])(cdrom_drive_t *) = { cdda, cd_extra, hidden_track, mixed_mode_cd };

  cdio_log_set_handler (log_handler);

  while(files[i] != NULL){
    rc = perform_test("/src/external-vcs/github/rocky/libcdio-paranoia/test/data/", files[i], funcs[i]);
    if(rc)
      return rc;
    i++;
  }

  return 0;
}
